unit IWExtCtrlsWAP;

interface

uses
  {$IFDEF VSNET}
  System.ComponentModel,
  System.Drawing.Design,
  IWNetClasses,
  System.Drawing, System.IO, System.Drawing.Imaging,
  {$ELSE}
  Classes,
    {$IFDEF Linux}
    IWCLXComponent, IWCLXClasses,
    {$ELSE}
    IWVCLComponent, IWVCLClasses,
    {$ENDIF}
  {$ENDIF}
  {$IFDEF Linux}QGraphics, {$ELSE}Graphics, {$ENDIF}
  SysUtils,
  IWBaseControl, IWControlWAP, IWTypes, IWFileReference, IWWAPTag, IWGlobal,
  IWRenderContext, IWJpegOptions, IWBaseInterfaces, IWBaseWAPControl,
  IWWAPRenderContext, IWUtils;

type
  TWAPImgAlign = (twiaTop, twiaBottom, twiaMiddle);

  TIWCustomImageWAP = class(TIWCustomControlWAP)
  protected
    FAltText: string;
    FPicture: TIWPicture;
    FUseSize: Boolean;
    FVSpace: string;
    FHSpace: string;
    FLocalsrc: string;
    FAlign: TWAPImgAlign;
    {$IFDEF VSNET}
    FImage: Image;
    {$ENDIF}
    //
    function CanAutoSize(var VNewWidth: Integer;
      var VNewHeight: Integer): Boolean; {$IFNDEF Linux} override; {$ENDIF}
    procedure CreatePicture;
    // Necessary because Borland made FOnClick protected
    function ImageWAP(const ASrc: string; AContext: TIWBaseWAPComponentContext): TIWWAPTag;
    procedure PictureChanged(ASender: TObject);
    procedure SetPicture(AValue: TIWPicture);
    {$IFDEF VSNET}
    procedure SetImage(AValue: Image);
    {$ENDIF}
    procedure InitControl; override;
    procedure SetVSpace(AValue: String);
    procedure SetHSpace(AValue: String);
    procedure SetLocalSrc(AValue: String);
    procedure SetAlign(const Value: TWAPImgAlign);
    {$IFDEF VSNET}
    function ShouldSerializeImage: Boolean;
    [Bindable(true)]    
    property Image: Image read FImage write SetImage;
    {$ENDIF}
  {$IFDEF CLR}
  strict protected
  {$ELSE}
  protected
  {$ENDIF}
    procedure Dispose(ADispose: Boolean); override;
  public
    property Picture: TIWPicture read FPicture write SetPicture;
    property Localsrc: string read FLocalsrc write SetLocalSrc;
  published
    {$IFDEF VSNET}
    [DefaultValue('')]
    {$ENDIF}
    property AltText: string read FAltText write FAltText;
    {$IFDEF VSNET}
    [DefaultValue('')]
    {$ENDIF}
    property VSpace: string read FVSpace write SetVSpace;
    {$IFDEF VSNET}
    [DefaultValue('')]
    {$ENDIF}
    property HSpace: string read FHSpace write SetHSpace;
    property Align: TWAPImgAlign read FAlign write SetAlign;
    {$IFNDEF Linux}
    // Necessary as Delphi defaults to False and causes streaming problems

    property AutoSize default True;
    {$ENDIF}
    property ExtraTagParams;

    property UseSize: Boolean read FUseSize write FUseSize;
  end;

  TIWDynamicImageWAP = class(TIWCustomImageWAP)
  protected
    FJpegOptions: TIWJpegOptions;
    procedure SetJpegOptions(const AValue: TIWJpegOptions);
    procedure InitControl; override;
  {$IFDEF CLR}
  strict protected
  {$ELSE}
  protected
  {$ENDIF}
    procedure Dispose(ADispose: Boolean); override;
  public
    function RenderWap(AContext: TIWBaseWapComponentContext): TIWWapTag; override;
  published
    property JpegOptions: TIWJpegOptions read FJpegOptions write SetJpegOptions;
  end;

  {$IFDEF VSNET}
  {$R icons\Atozed.Intraweb.TIWImageWAP.bmp}
  TIWImageWAP = class;
  [ToolboxItem(true), ToolboxBitmap(typeof(TIWImageWAP), 'TIWImageWAP.bmp')]
  {$ENDIF}
  TIWImageWAP = class(TIWDynamicImageWAP)
  published
    {$IFDEF VSNET}
    property Image;
    {$ELSE}
    property Picture;
    {$ENDIF}
    property Localsrc;
  end;

  {$IFDEF VSNET}
  {$R icons\Atozed.Intraweb.TIWImageFileWAP.bmp}
  TIWImageFileWAP = class;
  [ToolboxItem(true), ToolboxBitmap(typeof(TIWImageFileWAP), 'TIWImageFileWAP.bmp')]
  {$ENDIF}
  TIWImageFileWAP = class(TIWCustomImageWAP)
  protected
    FImageFile: TIWFileReference;
    //
    procedure ReferenceChanged(ASender: TObject);
    procedure SetImageFile(const AValue: TIWFileReference);

    procedure InitDesignTime; override;
    procedure InitControl; override;
  {$IFDEF CLR}
  strict protected
  {$ELSE}
  protected
  {$ENDIF}
    procedure Dispose(ADispose: Boolean); override;    
  public
    function RenderWap(AContext: TIWBaseWapComponentContext): TIWWapTag; override;
  published
    {$IFDEF VSNET}
    [DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]
    [Bindable(true)]    
    {$ENDIF}
    property ImageFile: TIWFileReference read FImageFile write SetImageFile;
  end;

implementation

uses
  {$IFDEF VSNET}
  Classes,
  {$ENDIF}
  IWServerControllerBase, SWSystem;

{ TIWCustomImageWAP }

function TIWCustomImageWAP.CanAutoSize(var VNewWidth,
  VNewHeight: Integer): Boolean;
begin
  Result := False;
  // FPicture is nil during initial create at design time for TIWImageFile (inherited Create)
  // Seems wierd to check AutoSize esp during streaming, but its required otehrwise during
  // AutoSize = False and streaming at runtime we get witdh and height of 0.
  if (FPicture <> nil) and (IsDesignMode = False) and (AutoSize = True) then begin
    if (FPicture.Width > 0) and (FPicture.Height > 0) then begin
      Result := True;
    end;
  end;
end;

procedure TIWCustomImageWAP.InitControl;
begin
  inherited;
  AutoSize := True;
  FUseSize := True;
  Width := 89;
  Height := 112;
  FAlign:= twiaBottom;
  FAltText := '';
  FVSpace := '';
  FHSpace := '';
end;

procedure TIWCustomImageWAP.CreatePicture;
begin
  FPicture := TIWPicture.Create;
  FPicture.OnChange := PictureChanged;
end;

procedure TIWCustomImageWAP.Dispose(ADispose: Boolean);
begin
  FreeAndNil(FPicture);
  inherited;
end;

function TIWCustomImageWAP.ImageWAP(const ASrc: string; AContext: TIWBaseWAPComponentContext): TIWWAPTag;
begin
  if AContext.PageContext.OutputTagType = wotWML then begin
    Result := TIWWAPTag.CreateTag('img');
    try
      Result.AddStringParam('src', ASrc);
      Result.AddStringParam('localsrc', Localsrc);
      if AltText = '' then begin
        AltText := 'imagetext';
      end;
      Result.AddStringParam('alt', AltText);
      if UseSize then begin
        Result.AddIntegerParam('width', Width);
        Result.AddIntegerParam('height', Height);
        // ??? Not sure adding vspace and hspace should be inside "if UseSize"
        Result.AddStringParam('vspace', VSpace);
        Result.AddStringParam('hspace', HSpace);
      end;
      case Align of
        twiaTop:
          Result.AddStringParam('align', 'top');
        twiaBottom:
          Result.AddStringParam('align', 'bottom');
        twiaMiddle:
          Result.AddStringParam('align', 'middle');
      end;
    except
      FreeAndNil(Result);
      raise;
    end;
  end else begin
    Result := TIWWAPTag.CreateTag('img');
    try
      Result.AddStringParam('src', ASrc);
      //Result.AddStringParam('name', Name);
      if AltText = '' then begin
        AltText := 'imagetext';
      end;
      Result.AddStringParam('alt', AltText);
      if UseSize then begin
        Result.AddIntegerParam('width', Width);
        Result.AddIntegerParam('height', Height);
      end;
    except
      FreeAndNil(Result);
      raise;
    end;
  end;
end;

procedure TIWCustomImageWAP.PictureChanged(ASender: TObject);
begin
  if AutoSize and (FPicture.Width > 0) and (FPicture.Height > 0) then begin
    SetBounds(Left, Top, FPicture.Width, FPicture.Height);
  end;
  Invalidate;
end;

{$IFDEF VSNET}
function TIWCustomImageWAP.ShouldSerializeImage: Boolean;
begin
  result := Assigned(FImage);
end;

procedure TIWCustomImageWAP.SetImage(AValue: Image);
Var
  ms: MemoryStream;
  ms1: TCLRStreamWrapper;
begin
  FImage := AValue;
  if Assigned(FImage) then begin
    ms := MemoryStream.Create; try
      FImage.Save(ms, ImageFormat.Bmp);
      ms1 := TCLRStreamWrapper.Create(ms); try
        FPicture.Graphic := TBitmap.Create;
        ms1.Position := 0;
        FPicture.Graphic.LoadFromStream(ms1);
      finally
        ms1.Free;
      end;
    finally
      ms.Free;
    end;
  end else begin
    FPicture.Graphic := nil;
  end;
end;
{$ENDIF}

procedure TIWCustomImageWAP.SetPicture(AValue: TIWPicture);
begin
  FPicture.Assign(AValue);
end;

procedure TIWCustomImageWAP.SetLocalSrc(AValue: String);
begin
  if FLocalSrc<>AValue then begin
     FLocalSrc:= AValue;
  end;
end;

procedure TIWCustomImageWAP.SetVSpace(AValue: String);
begin
  if FVSpace<>AValue then begin
     FVSpace:= AValue;
  end;
end;

procedure TIWCustomImageWAP.SetHSpace(AValue: String);
begin
  if FHSpace<>AValue then begin
     FHSpace:= AValue;
  end;
end;

procedure TIWCustomImageWAP.SetAlign(const Value: TWAPImgAlign);
begin
  if FAlign <> Value then begin
    FAlign := Value;
    Invalidate;
  end;
end;


{ TIWDynamicImageWAP }

procedure TIWDynamicImageWAP.InitControl;
begin
  inherited;
  CreatePicture;
  FJpegOptions := IWJpegOptions.TIWJpegOptions.Create;
end;

procedure TIWDynamicImageWAP.Dispose(ADispose: Boolean);
begin
  FreeAndNil(FJPegOptions);
  inherited;
end;


function TIWDynamicImageWAP.RenderWAP(AContext: TIWBaseWAPComponentContext): TIWWAPTag;
var
  LPathname: string;
  LPictureValid: Boolean;
  LBitmap: TBitmap;
begin
  Result := nil;
  LPathname := TIWServerControllerBase.NewCacheFile('jpg', true);
  // Is nil if no picture. Ex: Drop a TIWImage on form, dont load an image, run.
  LPictureValid:= (FPicture.Graphic <> nil) and not FPicture.Graphic.Empty;
  if LPictureValid or ( Trim(LocalSrc) <> '' ) then begin
      if LPictureValid then begin
        if AContext.PageContext.OutputTagType = wotWML then begin
          {$IFNDEF CLR}
          LBitmap := TBitmap.Create;
          try
            LBitmap.Assign(FPicture.Graphic);
            SaveAsWBmpFile(LBitmap,LPathname);
          finally
            FreeandNil(LBitmap);
          end;
          {$ENDIF}
        end else begin
           ToJPegFile(FPicture.Graphic, LPathname, FJPegOptions);
        end;
        Result := ImageWAP(GServerController.UserCacheURL + ExtractFilename(LPathname), AContext);
      end
      else begin
        Result := ImageWAP('', AContext);
      end;
  end
end;

procedure TIWDynamicImageWAP.SetJpegOptions(const AValue: TIWJpegOptions);
begin
  FJpegOptions.Assign(AValue);
end;


{ TIWImageFileWAP }

procedure TIWImageFileWAP.InitDesignTime;
begin
  inherited;
  if IsDesignMode then begin
     CreatePicture;
  end;
end;

procedure TIWImageFileWAP.InitControl;
begin
  inherited;
  FImageFile := TIWFileReference.Create;
  FImageFile.OnChange := ReferenceChanged;
end;

procedure TIWImageFileWAP.Dispose(ADispose: Boolean);
begin
  FreeAndNil(FImageFile);
  inherited;
end;

procedure TIWImageFileWAP.ReferenceChanged(ASender: TObject);
begin
  if IsDesignMode then begin
    if FileExists(ImageFile.Filename) then begin
      if LowerCase(ExtractFileExt(ImageFile.Filename)) = 'wbmp' then begin
        try
          {$IFNDEF CLR}
          LoadFromWBmpFile(FPicture.Bitmap,ImageFile.Filename);
          {$ENDIF}
        except
          FPicture.Graphic := nil;
        end;
      end else if LowerCase(ExtractFileExt(ImageFile.Filename)) = 'bmp' then begin
        try
          FPicture.LoadFromFile(ImageFile.Filename);
        except
          FPicture.Graphic := nil;
        end;
      end;
    end else begin
      FPicture.Graphic := nil;
    end;
    Invalidate;
  end else begin
    if AutoSize and (FPicture <> nil) then begin
      if (FPicture.Graphic.Width > 0) and (FPicture.Graphic.Height > 0) then begin
        Width := FPicture.Graphic.Width;
        Height := FPicture.Graphic.Height;
      end;
    end;
  end;
end;

function TIWImageFileWAP.RenderWAP(AContext: TIWBaseWAPComponentContext): TIWWAPTag;
begin
  Result := ImageWAP(ImageFile.Location(GServerController.FilesURL),AContext);
end;

procedure TIWImageFileWAP.SetImageFile(const AValue: TIWFileReference);
begin
  // Assign calls DoChange
  FImageFile.Assign(AValue);
end;

end.








